const products = [
    {
        id: 1,
        name: "Сухой корм для кошек Premium",
        price: 1200,
        category: "cats",
        description: "Сбалансированный корм для взрослых кошек с курицей и лососем",
        brand: "royal",
        badge: "Хит",
        image: "images/rocat.jpg"
    },
    {
        id: 2,
        name: "Игрушка-удочка для кошек",
        price: 350,
        category: "cats",
        description: "Интерактивная игрушка с перьями и погремушкой",
        brand: "whiskas",
        badge: "Новинка",
        image: "images/ydoch.webp"
    },
    {
        id: 3,
        name: "Корм для собак суперпремиум",
        price: 1500,
        category: "dogs",
        description: "Полнорационный корм для активных собак крупных пород",
        brand: "purina",
        badge: null,
        image: "images/rodog.jpeg"
    },
    {
        id: 4,
        name: "Ошейник кожаный с гравировкой",
        price: 450,
        category: "dogs",
        description: "Прочный кожаный ошейник с возможностью гравировки",
        brand: "royal",
        badge: null,
        image: "images/ochen.jpg"
    },
    {
        id: 5,
        name: "Колесо для хомяка",
        price: 600,
        category: "rodents",
        description: "Бесшумное колесо для грызунов диаметром 20 см",
        brand: "whiskas",
        badge: null,
        image: "images/kol.webp"
    },
    {
        id: 6,
        name: "Корм для попугаев Vitakraft",
        price: 300,
        category: "birds",
        description: "Смесь зерен и витаминов для попугаев средних пород",
        brand: "vitakraft",
        badge: null,
        image: "images/popu.jpg"
    },
    {
        id: 7,
        name: "Клетка для птиц большая",
        price: 2000,
        category: "birds",
        description: "Просторная клетка с аксессуарами и поддоном",
        brand: "purina",
        badge: "Акция",
        image: "images/klet.webp"
    },
    {
        id: 8,
        name: "Корм для рыб тропический",
        price: 250,
        category: "fish",
        description: "Специальный корм для тропических рыб в хлопьях",
        brand: "royal",
        badge: null,
        image: "images/tetra.jpg"
    },
    {
        id: 9,
        name: "Лакомства для собак с уткой",
        price: 520,
        category: "dogs",
        description: "Натуральные лакомства для дрессировки и поощрения",
        brand: "purina",
        badge: "Новинка",
        image: "images/ytka.webp"
    },
    {
        id: 10,
        name: "Домик для кошек плюшевый",
        price: 1800,
        category: "cats",
        description: "Мягкий двухуровневый домик с когтеточкой",
        brand: "royal",
        badge: "Новинка",
        image: "images/dom.webp"
    },
    {
        id: 11,
        name: "Игрушка для попугаев кольца",
        price: 270,
        category: "birds",
        description: "Развлекательная подвесная игрушка с колокольчиком",
        brand: "whiskas",
        badge: "Новинка",
        image: "images/popu.jpg"
    },
    {
        id: 12,
        name: "Аквариумный декор кораллы",
        price: 750,
        category: "fish",
        description: "Яркий безопасный декор для аквариумов до 60 литров",
        brand: "purina",
        badge: "Новинка",
        image: "images/korl.webp"
    }
];

let cart = [];
const cartSubscribers = [];

const demoCartItems = [
    {
        id: 1,
        name: "Сухой корм для кошек Premium",
        price: 1200,
        quantity: 2,
        category: "cats"
    },
    {
        id: 2,
        name: "Игрушка-удочка для кошек",
        price: 350,
        quantity: 1,
        category: "cats"
    },
    {
        id: 9,
        name: "Лакомства для собак с уткой",
        price: 520,
        quantity: 3,
        category: "dogs"
    }
];

let currentSlide = 0;
let slideInterval;

document.addEventListener('DOMContentLoaded', function() {
    initializeApp();
});

function initializeApp() {
    updateCartCount();
    setupEventListeners();
    
    if (document.querySelector('.hero-slider')) {
        initSlider();
    }
    
    if (document.getElementById('popular-products')) {
        loadPopularProducts();
    }

    initBrandsCarousel();
    
    if (document.getElementById('catalog-products')) {
        loadCatalogProducts();
        setupFilters();
    }
    
    setupSearch();

    if (document.getElementById('product-detail')) {
        loadProductDetail();
    }

    if (document.getElementById('cart-page')) {
        initializeCartPage();
    }

    setupFAQ();
}

function subscribeToCartUpdates(callback) {
    if (typeof callback === 'function') {
        cartSubscribers.push(callback);
    }
}

function notifyCartSubscribers() {
    cartSubscribers.forEach(callback => {
        try {
            callback();
        } catch (error) {
            console.error('Cart subscriber error:', error);
        }
    });
}

function setupEventListeners() {
    const menuBtn = document.getElementById('menu-btn');
    const mobileMenu = document.getElementById('mobile-menu');
    const closeMenu = document.getElementById('close-menu');
    
    if (menuBtn && mobileMenu) {
        menuBtn.addEventListener('click', () => {
            mobileMenu.classList.add('active');
        });
        
        closeMenu.addEventListener('click', () => {
            mobileMenu.classList.remove('active');
        });
    }
}

function initSlider() {
    const slides = document.querySelectorAll('.slide');
    const dotsContainer = document.querySelector('.slider-dots');
    const prevBtn = document.querySelector('.slider-prev');
    const nextBtn = document.querySelector('.slider-next');
    
    if (slides.length > 0) {
        slides[0].classList.add('active');
        goToSlide(0);
    }
    
    slides.forEach((slide, index) => {
        slide.addEventListener('click', () => goToSlide(index));
    });
    
    slides.forEach((_, index) => {
        const dot = document.createElement('div');
        dot.className = 'slider-dot';
        if (index === 0) dot.classList.add('active');
        dot.addEventListener('click', () => goToSlide(index));
        dotsContainer.appendChild(dot);
    });
    
    prevBtn.addEventListener('click', prevSlide);
    nextBtn.addEventListener('click', nextSlide);
    
    startAutoSlide();
    
    const slider = document.querySelector('.hero-slider');
    slider.addEventListener('mouseenter', stopAutoSlide);
    slider.addEventListener('mouseleave', startAutoSlide);
}

function goToSlide(index) {
    const slides = document.querySelectorAll('.slide');
    const dots = document.querySelectorAll('.slider-dot');
    const sliderContainer = document.querySelector('.slider-container');
    
    if (dots[currentSlide]) {
        dots[currentSlide].classList.remove('active');
    }
    
    if (slides[currentSlide]) {
        slides[currentSlide].classList.remove('active');
    }
    
    currentSlide = index;
    
    if (slides[currentSlide]) {
        slides[currentSlide].classList.add('active');
    }
    
    if (sliderContainer) {
        const translateX = -(currentSlide * 33.333);
        sliderContainer.style.transform = `translateX(${translateX}%)`;
    }
    
    if (dots[currentSlide]) {
        dots[currentSlide].classList.add('active');
    }
}

function nextSlide() {
    const slides = document.querySelectorAll('.slide');
    goToSlide((currentSlide + 1) % slides.length);
}

function prevSlide() {
    const slides = document.querySelectorAll('.slide');
    goToSlide((currentSlide - 1 + slides.length) % slides.length);
}

function startAutoSlide() {
    slideInterval = setInterval(nextSlide, 5000);
}

function stopAutoSlide() {
    clearInterval(slideInterval);
}

function initPopularCarousel() {
    const track = document.getElementById('popular-products');
    const prev = document.querySelector('.popular-prev');
    const next = document.querySelector('.popular-next');
    if (!track || !prev || !next) return;

    const getStep = () => {
        const card = track.querySelector('.product-card');
        if (!card) return 300;
        const style = getComputedStyle(track);
        const gap = parseInt(style.gap || '16');
        return card.offsetWidth + gap;
    };

    prev.addEventListener('click', () => {
        track.scrollBy({ left: -getStep(), behavior: 'smooth' });
    });
    next.addEventListener('click', () => {
        track.scrollBy({ left: getStep(), behavior: 'smooth' });
    });
}

function initBrandsCarousel() {
    const track = document.getElementById('brands-track');
    const prev = document.querySelector('.brands-prev');
    const next = document.querySelector('.brands-next');

    if (!track || !prev || !next) return;

    const getGap = () => {
        const style = getComputedStyle(track);
        const gapValue = style.gap || style.columnGap || '16px';
        const parsed = parseInt(gapValue, 10);
        return isNaN(parsed) ? 16 : parsed;
    };

    const getStep = () => {
        const card = track.querySelector('.brand-card');
        if (!card) return 200;
        return card.offsetWidth + getGap();
    };

    prev.addEventListener('click', () => {
        track.scrollBy({ left: -getStep(), behavior: 'smooth' });
    });

    next.addEventListener('click', () => {
        track.scrollBy({ left: getStep(), behavior: 'smooth' });
    });

    track.addEventListener('wheel', event => {
        if (Math.abs(event.deltaY) > Math.abs(event.deltaX)) {
            event.preventDefault();
            track.scrollBy({ left: event.deltaY, behavior: 'smooth' });
        }
    }, { passive: false });
}

function loadPopularProducts() {
    const container = document.getElementById('popular-products');
    const newArrivals = products.filter(product => product.badge === 'Новинка');
    const restProducts = products.filter(product => product.badge !== 'Новинка');
    const popularProducts = [...newArrivals, ...restProducts].slice(0, 6);
    
    container.innerHTML = popularProducts.map(product => `
        <div class="product-card">
            <div class="product-image">
                ${product.image ? `<img src="${product.image}" alt="${product.name}" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"><i class="fas fa-paw" style="display: none;"></i>` : '<i class="fas fa-paw"></i>'}
                ${product.badge ? `<span class="product-badge">${product.badge}</span>` : ''}
            </div>
            <div class="product-info">
                <div class="product-category">${getCategoryName(product.category)}</div>
                <h3>${product.name}</h3>
                <p class="product-description">${product.description}</p>
                <div class="product-footer">
                    <div class="product-price">${product.price} ₽</div>
                    <div class="product-actions">
                        <a href="product.html?id=${product.id}" class="product-detail-link">Подробнее</a>
                        <button class="add-to-cart" data-id="${product.id}">
                            <i class="fas fa-shopping-cart"></i>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `).join('');

    container.querySelectorAll('.add-to-cart').forEach(button => {
        button.addEventListener('click', addToCart);
    });

    initPopularCarousel();
}

function loadCatalogProducts(productsToShow = products) {
    const container = document.getElementById('catalog-products');
    
    container.innerHTML = productsToShow.map(product => `
        <div class="product-card">
            <div class="product-image">
                ${product.image ? `<img src="${product.image}" alt="${product.name}" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"><i class="fas fa-paw" style="display: none;"></i>` : '<i class="fas fa-paw"></i>'}
                ${product.badge ? `<span class="product-badge">${product.badge}</span>` : ''}
            </div>
            <div class="product-info">
                <div class="product-category">${getCategoryName(product.category)}</div>
                <h3>${product.name}</h3>
                <p class="product-description">${product.description}</p>
                <div class="product-footer">
                    <div class="product-price">${product.price} ₽</div>
                    <div class="product-actions">
                        <a href="product.html?id=${product.id}" class="product-detail-link">Подробнее</a>
                        <button class="add-to-cart" data-id="${product.id}">
                            <i class="fas fa-shopping-cart"></i>
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `).join('');
    
    document.getElementById('products-count').textContent = productsToShow.length;
    
    container.querySelectorAll('.add-to-cart').forEach(button => {
        button.addEventListener('click', addToCart);
    });
}

function setupFilters() {
    const applyFiltersBtn = document.querySelector('.apply-filters');
    const resetFiltersBtn = document.querySelector('.reset-filters');
    const sortSelect = document.querySelector('.sort-select');
    
    if (applyFiltersBtn) {
        applyFiltersBtn.addEventListener('click', applyFilters);
    }
    
    if (resetFiltersBtn) {
        resetFiltersBtn.addEventListener('click', resetFilters);
    }
    
    if (sortSelect) {
        sortSelect.addEventListener('change', applyFilters);
    }
}

function applyFilters() {
    const categoryFilters = document.querySelectorAll('input[name="category"]:checked');
    const brandFilters = document.querySelectorAll('input[name="brand"]:checked');
    const priceSlider = document.getElementById('price-slider');
    const sortBy = document.querySelector('.sort-select').value;
    
    let filteredProducts = [...products];
    
    if (categoryFilters.length > 0) {
        const selectedCategories = Array.from(categoryFilters).map(cb => cb.value);
        filteredProducts = filteredProducts.filter(product => 
            selectedCategories.includes(product.category)
        );
    }

    if (brandFilters.length > 0) {
        const selectedBrands = Array.from(brandFilters).map(cb => cb.value);
        filteredProducts = filteredProducts.filter(product => 
            selectedBrands.includes(product.brand)
        );
    }
    
    const maxPrice = parseInt(priceSlider.value);
    filteredProducts = filteredProducts.filter(product => product.price <= maxPrice);
    
    switch(sortBy) {
        case 'price-low':
            filteredProducts.sort((a, b) => a.price - b.price);
            break;
        case 'price-high':
            filteredProducts.sort((a, b) => b.price - a.price);
            break;
        case 'new':
            filteredProducts.reverse();
            break;
        default:
            break;
    }
    
    loadCatalogProducts(filteredProducts);
}

function resetFilters() {
    document.querySelectorAll('input[type="checkbox"]').forEach(checkbox => {
        checkbox.checked = false;
    });
    
    document.getElementById('price-slider').value = 5000;
    
    document.querySelector('.sort-select').value = 'popular';
    
    loadCatalogProducts();
}

function setupSearch() {
    const searchInput = document.getElementById('search-input');
    const searchResults = document.getElementById('search-results');
    const searchButton = document.getElementById('search-btn');

    if (searchInput && searchResults) {
        const runSearch = () => {
            const query = searchInput.value.toLowerCase().trim();

            if (query.length === 0) {
                searchResults.innerHTML = '<div class="search-result-item">Введите запрос</div>';
                searchResults.classList.add('active');
                return;
            }

            const filteredProducts = products.filter(product =>
                product.name.toLowerCase().includes(query) ||
                product.description.toLowerCase().includes(query) ||
                getCategoryName(product.category).toLowerCase().includes(query)
            );

            displaySearchResults(filteredProducts, query);
        };

        searchInput.addEventListener('input', runSearch);

        if (searchButton) {
            searchButton.addEventListener('click', function(e) {
                e.preventDefault();
                searchInput.focus();
                runSearch();
            });
        }

        document.addEventListener('click', function(e) {
            if (!searchInput.contains(e.target) && !searchResults.contains(e.target)) {
                searchResults.classList.remove('active');
            }
        });
    }
}

function displaySearchResults(results, query) {
    const searchResults = document.getElementById('search-results');
    
    if (results.length === 0) {
        searchResults.innerHTML = '<div class="search-result-item">Ничего не найдено</div>';
    } else {
        searchResults.innerHTML = results.map(product => `
            <div class="search-result-item" data-id="${product.id}">
                <div style="font-weight: 600;">${highlightText(product.name, query)}</div>
                <div style="font-size: 0.9rem; color: #666;">${getCategoryName(product.category)} • ${product.price} ₽</div>
            </div>
        `).join('');
        
        searchResults.querySelectorAll('.search-result-item').forEach(item => {
            item.addEventListener('click', function() {
                const productId = parseInt(this.dataset.id);
                if (!isNaN(productId)) {
                    window.location.href = `product.html?id=${productId}`;
                }
            });
        });
    }
    
    searchResults.classList.add('active');
}

function highlightText(text, query) {
    if (!query) return text;
    const regex = new RegExp(`(${query})`, 'gi');
    return text.replace(regex, '<mark>$1</mark>');
}

function setupFAQ() {
    const faqList = document.getElementById('faq-list');
    if (!faqList) return;
    faqList.querySelectorAll('.faq-item').forEach(item => {
        const btn = item.querySelector('.faq-question');
        if (btn) {
            btn.addEventListener('click', () => {
                item.classList.toggle('active');
            });
        }
    });
}

function loadProductDetail() {
    const detailContainer = document.getElementById('product-detail');
    if (!detailContainer) return;

    const params = new URLSearchParams(window.location.search);
    const productId = parseInt(params.get('id'), 10);
    const product = products.find(p => p.id === productId);

    if (!product) {
        detailContainer.innerHTML = `
            <div class="product-detail-empty">
                <h2>Товар не найден</h2>
                <p>Возможно, он был удалён или временно недоступен.</p>
                <a href="catalog.html" class="btn btn-primary">Вернуться в каталог</a>
            </div>
        `;
        return;
    }

    document.title = `${product.name} - Зоорай`;

    detailContainer.innerHTML = `
        <div class="product-detail-card">
            <div class="product-detail-media">
                <img src="${product.image || 'images/default.jpg'}" alt="${product.name}" style="width: 100%; height: 100%; object-fit: cover; border-radius: var(--radius-lg);" onerror="this.onerror=null; this.style.display='none'; this.nextElementSibling.style.display='flex';">
                <div class="product-detail-icon" style="display: none;">
                    <i class="fas fa-paw"></i>
                </div>
                ${product.badge ? `<span class="product-badge">${product.badge}</span>` : ''}
            </div>
            <div class="product-detail-info">
                <div class="product-detail-category">${getCategoryName(product.category)}</div>
                <h1>${product.name}</h1>
                <p class="product-detail-description">${product.description}. Подходит для ежедневного использования. Рекомендации по применению и хранению доступны ниже.</p>

                <div class="features-grid" style="display:grid;grid-template-columns:repeat(auto-fit,minmax(180px,1fr));gap:1rem;margin-bottom:1.5rem;">
                    <div class="feature-card" style="padding:1rem;background:var(--primary-light);border-radius:8px;">
                        <strong>Бренд:</strong> ${product.brand ? product.brand.toUpperCase() : '—'}
                    </div>
                    <div class="feature-card" style="padding:1rem;background:var(--primary-light);border-radius:8px;">
                        <strong>Категория:</strong> ${getCategoryName(product.category)}
                    </div>
                    <div class="feature-card" style="padding:1rem;background:var(--primary-light);border-radius:8px;">
                        <strong>Доставка:</strong> от 1 дня
                    </div>
                    <div class="feature-card" style="padding:1rem;background:var(--primary-light);border-radius:8px;">
                        <strong>Гарантия:</strong> 14 дней
                    </div>
                </div>

                <div class="product-detail-price">${product.price} ₽</div>
                <div class="product-detail-actions">
                    <button class="btn btn-primary product-detail-add" data-id="${product.id}">
                        <i class="fas fa-shopping-cart"></i>
                        Добавить в корзину
                    </button>
                    <a href="cart.html" class="btn btn-outline">Перейти в корзину</a>
                </div>

                <div class="product-detail-description" style="margin-top:1.5rem;">
                    Подробности: изготовлено из качественных материалов. Соответствует стандартам безопасности. Идеально для питомцев соответствующей категории. Представленные характеристики носят ознакомительный характер и могут различаться в зависимости от партии поставки.
                </div>
            </div>
        </div>
    `;

    const addButton = detailContainer.querySelector('.product-detail-add');
    if (addButton) {
        addButton.addEventListener('click', addToCart);
    }
}

function initializeCartPage() {
    const itemsContainer = document.getElementById('cart-page-items');
    const emptyState = document.getElementById('cart-page-empty');
    const summarySection = document.getElementById('cart-page-summary');
    const totalPriceElement = document.getElementById('cart-page-total');
    const checkoutBtn = document.getElementById('cart-page-checkout');

    if (!itemsContainer || !emptyState || !summarySection || !totalPriceElement || !checkoutBtn) {
        return;
    }

    const renderCartPage = () => {
        if (demoCartItems.length === 0) {
            itemsContainer.style.display = 'none';
            summarySection.style.display = 'none';
            emptyState.style.display = 'block';
            return;
        }

        itemsContainer.style.display = 'grid';
        summarySection.style.display = 'block';
        emptyState.style.display = 'none';

        const total = demoCartItems.reduce((sum, item) => sum + (item.price * item.quantity), 0);
        const totalItems = demoCartItems.reduce((sum, item) => sum + item.quantity, 0);
        totalPriceElement.textContent = `${total} ₽`;

        summarySection.querySelector('.cart-page-count').textContent = `${totalItems} ${getWordByNumber(totalItems, 'товар', 'товара', 'товаров')}`;

        itemsContainer.innerHTML = demoCartItems.map(item => {
            const product = products.find(p => p.id === item.id);
            return `
            <div class="cart-page-item">
                <div class="cart-item-image">
                    ${product && product.image ? `<img src="${product.image}" alt="${product.name}" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"><i class="fas fa-paw" style="display: none;"></i>` : '<i class="fas fa-paw"></i>'}
                </div>
                <div class="cart-item-info">
                    <div class="cart-item-name">${item.name}</div>
                    <div class="cart-item-price">${item.price} ₽</div>
                </div>
                <div class="cart-item-actions">
                    <button class="quantity-btn minus" data-id="${item.id}">-</button>
                    <span>${item.quantity}</span>
                    <button class="quantity-btn plus" data-id="${item.id}">+</button>
                </div>
                <button class="remove-item" data-id="${item.id}">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
            `;
        }).join('');

        itemsContainer.querySelectorAll('.quantity-btn.plus').forEach(btn => {
            btn.addEventListener('click', function() {
                const id = parseInt(this.dataset.id);
                showNotification("Функция изменения количества недоступна в демо-режиме");
            });
        });

        itemsContainer.querySelectorAll('.quantity-btn.minus').forEach(btn => {
            btn.addEventListener('click', function() {
                const id = parseInt(this.dataset.id);
                showNotification("Функция изменения количества недоступна в демо-режиме");
            });
        });

        itemsContainer.querySelectorAll('.remove-item').forEach(btn => {
            btn.addEventListener('click', function() {
                const id = parseInt(this.dataset.id);
                showNotification("Функция удаления товара недоступна в демо-режиме");
            });
        });
    };

    checkoutBtn.addEventListener('click', () => {
        const total = demoCartItems.reduce((sum, item) => sum + (item.price * item.quantity), 0);
        showNotification(`Демо-заказ оформлен! Сумма: ${total} ₽ (демо-режим)`);
    });

    renderCartPage();
}

function getWordByNumber(number, one, few, many) {
    const n = Math.abs(number) % 100;
    const n1 = n % 10;

    if (n > 10 && n < 20) return many;
    if (n1 > 1 && n1 < 5) return few;
    if (n1 === 1) return one;
    return many;
}

function addToCart(event) {
    const productId = parseInt(event.currentTarget.dataset.id);
    const product = products.find(p => p.id === productId);
    
    if (product) {
        showNotification(`Товар "${product.name}" добавлен в корзину! (демо-режим)`);
    }
    
    return;
}

function updateCartCount() {
    const cartCount = document.querySelector('.cart-count');
    if (cartCount) {
        const totalItems = demoCartItems.reduce((sum, item) => sum + item.quantity, 0);
        cartCount.textContent = totalItems;
    }
}

function updateCartItemQuantity(productId, change) {
    showNotification("Функция изменения количества недоступна в демо-режиме");
    return;
}

function removeCartItem(productId) {
    showNotification("Функция удаления товара недоступна в демо-режиме");
    return;
}

function getCategoryName(category) {
    const categories = {
        'cats': 'Для кошек',
        'dogs': 'Для собак',
        'rodents': 'Для грызунов',
        'birds': 'Для птиц',
        'fish': 'Для рыб'
    };
    return categories[category] || category;
}

function showNotification(message) {
    const notification = document.createElement('div');
    notification.className = 'notification';
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas fa-check-circle"></i>
            <span>${message}</span>
        </div>
    `;
    
    notification.style.cssText = `
        position: fixed;
        top: 100px;
        right: 20px;
        background: var(--primary);
        color: white;
        padding: 1rem 1.5rem;
        border-radius: 8px;
        box-shadow: 0 10px 25px rgba(0,0,0,0.15);
        z-index: 10000;
        animation: slideIn 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 300);
    }, 3000);
}

const notificationStyles = `
@keyframes slideIn {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

@keyframes slideOut {
    from {
        transform: translateX(0);
        opacity: 1;
    }
    to {
        transform: translateX(100%);
        opacity: 0;
    }
}

.notification-content {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}
`;

const styleSheet = document.createElement('style');
styleSheet.textContent = notificationStyles;
document.head.appendChild(styleSheet);